#!/bin/bash
## Instalador Avirato PMS para Linux
## Actualizado: 2025-11-11
## Compatible con Ubuntu 20.04+, Linux Mint 20+, Debian 11+

# Verificar dependencias necesarias
for cmd in zenity unzip; do
    if ! command -v $cmd >/dev/null 2>&1; then
        echo "Error: $cmd no está instalado. Por favor instálelo primero."
        echo "En Ubuntu/Debian: sudo apt install $cmd"
        exit 1
    fi
done

NAME_APP="Avirato"
NAME_APP_REPAIR="AviratoRepair"
RUTA_ICONO_MODAL=`dirname $0`/".Icono.png"


FILE_LICENSE_FOLDER=`dirname $0`/".licencia"
JRE_NAME="JRE_17.0.15"
JRE_ZIP="$JRE_NAME.zip"
APP_NAME_DESKTOP="$NAME_APP-PMS.desktop"
APP_NAME_DESKTOP_REPAIR="$NAME_APP-Repair.desktop"

# Diálogo de idioma (una sola selección)
LANG_CHOICE=$(zenity --list \
  --title="Seleccionar idioma / Select language" \
  --text="Elige el idioma" \
  --width=420 --height=240 \
  --window-icon="$RUTA_ICONO_MODAL" \
  --column="Idioma / Language" \
  "Español" "English" "Français" "Italiano" "Português")

# Si cancela/cierra, salimos
if [ $? -ne 0 ] || [ -z "$LANG_CHOICE" ]; then
    echo "Cancelado por el usuario."
    exit 1
fi

## Mapeo idioma -> sufijo + textos
LICENSE_SUFFIX="es"
TITLE="Avirato Licencia"
CHECKBOX="He leído y acepto los términos."
TEXT_THANKS="¡Gracias por elegir Avirato!

Recuerde que estamos a su entera disposición en cualquiera de nuestras líneas telefónicas para ayudarle si fuese necesario.
Puede ponerse en contacto con nosotros en el teléfono +34-91 269 0123

Ante cualquier duda, por favor póngase en contacto con nosotros.

Avirato S.L."
case "$LANG_CHOICE" in
  English)
    LICENSE_SUFFIX="en"
    TITLE="Avirato Licence"
    CHECKBOX="I read and accept the terms."
    TEXT_THANKS="Thank you for choosing Avirato!

Please remember that we are fully at your disposal through any of our telephone lines should you need assistance.
You can contact us at +34-91 269 0123.

If you have any questions, please do not hesitate to contact us.

Avirato S.L."
    ;;
  Français)
    LICENSE_SUFFIX="fr"
    TITLE="Licence Avirato"
    CHECKBOX="J'ai lu et j'accepte les conditions."
    TEXT_THANKS="Merci d’avoir choisi Avirato !

N’oubliez pas que nous restons entièrement à votre disposition sur nos lignes téléphoniques pour vous aider si nécessaire.
Vous pouvez nous contacter au +34-91 269 0123.

Pour toute question, n’hésitez pas à nous contacter.

Avirato S.L."
    ;;
  Italiano)
    LICENSE_SUFFIX="it"
    TITLE="Licenza Avirato"
    CHECKBOX="Ho letto e accetto i termini."
    TEXT_THANKS="Grazie per aver scelto Avirato!

Ricordi che siamo sempre a Sua completa disposizione tramite le nostre linee telefoniche per offrirLe assistenza, se necessario.
Può contattarci al numero +34-91 269 0123.

Per qualsiasi dubbio, La invitiamo a mettersi in contatto con noi.

Avirato S.L."
    ;;
  Português)
    LICENSE_SUFFIX="pt"
    TITLE="Licença Avirato"
    CHECKBOX="Li e aceito os termos."
    TEXT_THANKS="Obrigado por escolher a Avirato!

Lembre-se de que estamos totalmente à sua disposição através das nossas linhas telefónicas para ajudá-lo, se necessário.
Pode contactar-nos pelo telefone +34-91 269 0123.

Em caso de dúvida, por favor não hesite em contactar-nos.

Avirato S.L."
    ;;
  Español|*)
    LICENSE_SUFFIX="es"
    TITLE="Avirato Licencia"
    CHECKBOX="He leído y acepto los términos."
    TEXT_THANKS="¡Gracias por elegir Avirato!

Recuerde que estamos a su entera disposición en cualquiera de nuestras líneas telefónicas para ayudarle si fuese necesario.
Puede ponerse en contacto con nosotros en el teléfono +34-91 269 0123

Ante cualquier duda, por favor póngase en contacto con nosotros.

Avirato S.L."
    ;;
esac

# Selección del fichero de licencia según idioma (fallback a licencia_en.txt)
FILE_LICENSE="$FILE_LICENSE_FOLDER/licencia_${LICENSE_SUFFIX}.txt"
echo "$FILE_LICENSE"
if [ ! -f "$FILE_LICENSE" ]; then
    FILE_LICENSE="$FILE_LICENSE_FOLDER/licencia_en.txt"
fi

# Mostrar licencia
zenity --text-info \
  --width="580" --height="450" \
  --window-icon="$RUTA_ICONO_MODAL" \
  --title="$TITLE" \
  --filename="$FILE_LICENSE" \
  --checkbox="$CHECKBOX"

#cancel -> 1
#aceptar -> 0
#cerrar -> -1

case $? in
	0) 
		# Licencia aceptada, ahora preguntar ubicación de instalación
		INSTALL_CHOICE=$(zenity --list \
		  --title="Ubicación de instalación / Installation Location" \
		  --text="Seleccione dónde desea instalar Avirato:\nSelect where you want to install Avirato:" \
		  --width=550 --height=280 \
		  --window-icon="$RUTA_ICONO_MODAL" \
		  --column="Opción / Option" \
		  "Instalación local (recomendado) - $HOME/Avirato" \
		  "Instalación del sistema - /opt/Avirato (requiere permisos / requires permissions)")
		
		# Si cancela, salimos
		if [ $? -ne 0 ] || [ -z "$INSTALL_CHOICE" ]; then
		    zenity --error --text="Instalación cancelada" --window-icon="$RUTA_ICONO_MODAL"
		    exit 1
		fi
		
		# Determinar rutas según elección
		if echo "$INSTALL_CHOICE" | grep -q "sistema"; then
		    INSTALL_DIR="/opt/$NAME_APP"
		    RUTA_APPLICATIONS="/usr/share/applications"
		    ICON_DIR="/usr/share/icons/hicolor/256x256/apps"
		    REQUIRES_SUDO=true
		else
		    INSTALL_DIR="$HOME/$NAME_APP"
		    RUTA_APPLICATIONS="$HOME/.local/share/applications"
		    ICON_DIR="$HOME/.local/share/icons/hicolor/256x256/apps"
		    REQUIRES_SUDO=false
		fi
		
		# Definir rutas finales
		RUTA_ARRANCAR_SH="$INSTALL_DIR/arrancar.sh"
		RUTA_JRE="$INSTALL_DIR/$JRE_NAME/bin/java"
		RUTA_AVIRATO_JAR="$INSTALL_DIR/$NAME_APP.jar"
		RUTA_AVIRATO_REPAIRT_JAR="$INSTALL_DIR/$NAME_APP_REPAIR.jar"
		RUTO_ICONO="$ICON_DIR/avirato.png"
		RUTO_ICONO_REPAIR="$ICON_DIR/avirato-repair.png"
		
		# Iniciar instalación con barra de progreso
		(
		echo "10" ; echo "# Creando directorios..."
		# Crear directorios necesarios
		if [ "$REQUIRES_SUDO" = true ]; then
			sudo mkdir -p "$INSTALL_DIR" "$ICON_DIR" "$RUTA_APPLICATIONS"
		else
			mkdir -p "$INSTALL_DIR" "$ICON_DIR" "$RUTA_APPLICATIONS"
		fi
		
		echo "20" ; echo "# Copiando archivos de Avirato..."
		# Copiar aplicación
		if [ "$REQUIRES_SUDO" = true ]; then
			sudo cp -rf "`dirname $0`/$NAME_APP"/* "$INSTALL_DIR/"
		else
			cp -rf "`dirname $0`/$NAME_APP"/* "$INSTALL_DIR/"
		fi

		echo "40" ; echo "# Instalando Java Runtime Environment..."
		# Descomprimir JRE (silenciosamente)
		FILEZIP="`dirname $0`/$JRE_ZIP"
		if [ "$REQUIRES_SUDO" = true ]; then
			sudo unzip -q -o "$FILEZIP" -d "$INSTALL_DIR"
		else
			unzip -q -o "$FILEZIP" -d "$INSTALL_DIR"
		fi
		
		echo "60" ; echo "# Instalando iconos..."
		# Copiar iconos a ubicación estándar
		if [ "$REQUIRES_SUDO" = true ]; then
			sudo cp "`dirname $0`/.Icono.png" "$ICON_DIR/avirato.png"
			sudo cp "`dirname $0`/.IconoRepair.png" "$ICON_DIR/avirato-repair.png"
		else
			cp "`dirname $0`/.Icono.png" "$ICON_DIR/avirato.png"
			cp "`dirname $0`/.IconoRepair.png" "$ICON_DIR/avirato-repair.png"
		fi
		
		echo "70" ; echo "# Creando scripts de arranque..."
		# Script principal (arrancar.sh) - crear en temporal primero
		TEMP_SCRIPT="/tmp/avirato_arrancar_$$.sh"
		cat > "$TEMP_SCRIPT" << 'EOFSCRIPT'
#!/bin/bash
cd "$(dirname "$0")"
EOFSCRIPT
		echo "$RUTA_JRE \\" >> "$TEMP_SCRIPT"
		echo "  -Djava.security.policy=applet.policy \\" >> "$TEMP_SCRIPT"
		echo "  -Dfile.encoding=UTF-8 \\" >> "$TEMP_SCRIPT"
		echo "  -Dsun.java2d.dpiaware=false \\" >> "$TEMP_SCRIPT"
		echo "  --add-modules=javafx.swing,javafx.graphics,javafx.fxml,javafx.media,javafx.web \\" >> "$TEMP_SCRIPT"
		echo "  --add-reads javafx.graphics=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-opens javafx.controls/com.sun.javafx.charts=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-opens javafx.graphics/com.sun.javafx.iio=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-opens javafx.graphics/com.sun.javafx.iio.common=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-opens javafx.graphics/com.sun.javafx.css=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-opens javafx.base/com.sun.javafx.runtime=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-exports=java.desktop/sun.swing=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-exports=java.desktop/sun.swing.table=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-exports=java.desktop/sun.swing.plaf.synth=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-opens=java.desktop/javax.swing.plaf.synth=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-opens=java.desktop/javax.swing.plaf.basic=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-opens=java.desktop/javax.swing=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-opens=java.desktop/javax.swing.tree=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-opens=java.desktop/java.awt.event=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-exports=java.desktop/sun.awt.shell=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-exports=java.base/sun.security.action=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-opens=java.base/java.lang=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-exports=java.base/java.lang=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-exports=java.desktop/sun.awt=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  --add-exports=java.desktop/sun.java2d=ALL-UNNAMED \\" >> "$TEMP_SCRIPT"
		echo "  -jar \"$RUTA_AVIRATO_JAR\"" >> "$TEMP_SCRIPT"
		
		# Mover el script a la ubicación final
		if [ "$REQUIRES_SUDO" = true ]; then
			sudo mv "$TEMP_SCRIPT" "$RUTA_ARRANCAR_SH"
			sudo chmod +x "$RUTA_ARRANCAR_SH"
		else
			mv "$TEMP_SCRIPT" "$RUTA_ARRANCAR_SH"
			chmod +x "$RUTA_ARRANCAR_SH"
		fi

		echo "80" ; echo "# Creando accesos directos..."
		# Acceso directo principal de Avirato
		TEMP_DESKTOP="/tmp/avirato-pms_$$.desktop"
		cat > "$TEMP_DESKTOP" << EOFDESKTOP
[Desktop Entry]
Version=1.0
Type=Application
Name=Avirato PMS
GenericName=Property Management System
Comment=Sistema de Gestión Hotelera - Avirato S.L.
Keywords=PMS;Avirato;Manager;Booking;Hotel;
Exec=$RUTA_ARRANCAR_SH
Icon=avirato
Terminal=false
Categories=Office;Finance;Economy;
StartupNotify=true
StartupWMClass=avirato.main
EOFDESKTOP
		
		# Acceso directo de Avirato Repair
		TEMP_DESKTOP_REPAIR="/tmp/avirato-repair_$$.desktop"
		cat > "$TEMP_DESKTOP_REPAIR" << EOFDESKTOP2
[Desktop Entry]
Version=1.0
Type=Application
Name=Avirato Repair
GenericName=Avirato Repair Tool
Comment=Herramienta de reparación - Avirato S.L.
Keywords=PMS;Avirato;Repair;Manager;
Exec=$RUTA_JRE -jar "$RUTA_AVIRATO_REPAIRT_JAR"
Icon=avirato-repair
Terminal=false
Categories=Office;Utility;
StartupNotify=true
EOFDESKTOP2
		
		# Mover archivos desktop a su ubicación
		if [ "$REQUIRES_SUDO" = true ]; then
			sudo mv "$TEMP_DESKTOP" "$RUTA_APPLICATIONS/$APP_NAME_DESKTOP"
			sudo mv "$TEMP_DESKTOP_REPAIR" "$RUTA_APPLICATIONS/$APP_NAME_DESKTOP_REPAIR"
			sudo chmod 644 "$RUTA_APPLICATIONS/$APP_NAME_DESKTOP"
			sudo chmod 644 "$RUTA_APPLICATIONS/$APP_NAME_DESKTOP_REPAIR"
		else
			mv "$TEMP_DESKTOP" "$RUTA_APPLICATIONS/$APP_NAME_DESKTOP"
			mv "$TEMP_DESKTOP_REPAIR" "$RUTA_APPLICATIONS/$APP_NAME_DESKTOP_REPAIR"
			chmod 644 "$RUTA_APPLICATIONS/$APP_NAME_DESKTOP"
			chmod 644 "$RUTA_APPLICATIONS/$APP_NAME_DESKTOP_REPAIR"
		fi
		
		echo "90" ; echo "# Configurando permisos..."
		# Configurar permisos finales
		if [ "$REQUIRES_SUDO" = true ]; then
			sudo chmod 755 "$RUTA_JRE"
			sudo chmod 644 "$ICON_DIR/avirato.png" "$ICON_DIR/avirato-repair.png"
		else
			chmod 755 "$RUTA_JRE"
			chmod 644 "$ICON_DIR/avirato.png" "$ICON_DIR/avirato-repair.png"
		fi
		
		echo "95" ; echo "# Actualizando sistema..."
		# Refrescar menus y cache de iconos
		if [ "$REQUIRES_SUDO" = false ]; then
			xdg-desktop-menu install "$RUTA_APPLICATIONS/$APP_NAME_DESKTOP" 2>/dev/null || true
			xdg-desktop-menu install "$RUTA_APPLICATIONS/$APP_NAME_DESKTOP_REPAIR" 2>/dev/null || true
		fi
		
		# Actualizar caché de iconos
		if [ "$REQUIRES_SUDO" = true ]; then
			sudo gtk-update-icon-cache -f -t /usr/share/icons/hicolor 2>/dev/null || true
		else
			gtk-update-icon-cache -f -t "$HOME/.local/share/icons/hicolor" 2>/dev/null || true
		fi
		
		echo "100" ; echo "# Instalación completada!"
		) | zenity --progress \
		  --title="Instalando Avirato PMS" \
		  --window-icon="$RUTA_ICONO_MODAL" \
		  --width=450 \
		  --percentage=0 \
		  --auto-close \
		  --no-cancel
		
		# Verificar si la instalación fue exitosa
		if [ $? -eq 0 ]; then
			zenity --info --width="480" --height="230" \
			  --window-icon="$RUTA_ICONO_MODAL" \
			  --title="Instalación completada" \
			  --text="$TEXT_THANKS"
			
			# Ejecutar Avirato en segundo plano
			nohup "$RUTA_ARRANCAR_SH" >/dev/null 2>&1 &
		else
			zenity --error --text="La instalación falló. Por favor, inténtelo de nuevo." \
			  --window-icon="$RUTA_ICONO_MODAL"
			exit 1
		fi
		;;
	1)
		zenity --info --text="Instalación cancelada" --window-icon="$RUTA_ICONO_MODAL"
		exit 0
		;;
esac

